<?php
/*
 * Computer Magic Tool Kit (CMTK)
 * Copyright 2006 Computer Magic and Software Design.
 * All rights reserved.
 */

/*
 * CMTK Page Object - Process and render the page
 * Processing Steps:
 * 1 - Create the page object and load its meta data from the database
 *  using the current URL
 * 2 - Load the template for this page and store it for use later
 * 3 - Start output buffering
 * 3 - Physical pages are allowed to run, output is buffered
 * 4 - Physical page ends and RenderPage is called by ob_start callback
 * 5 - RenderPage puts template/page text together and processes tags
 * 6 - Final page html is sent back to the browser
 */

include_once "user.php";

/*
 * This class defines a page object and allows it to interact with
 * its various parts.
 */
class CMTKPage extends CMTKPersistentObject {
    protected $PARTS = array();    

    static function GetCurrentURL() {
        //TODO: if this is running in a sub folder,
        // strip the url down to the app folder so it matches properly
        return $_SERVER["SCRIPT_NAME"];
    }

    function __construct($page_url = "") {
        // Load the page info from the database
        CMTKEvent::FireEvent("CMTKPage_PreLoad", $this);        
        $this->SetTableName("cmtk_pages");
        $this->SetIDField("page_url");
        if ($page_url == "") {
            $this->SetIDValue(CMTKPage::GetCurrentURL());
        } else {
            $this->SetIDValue($page_url);
        }        
        $this->LoadObjectInfo();
        if (cmtk_GetConfigValue("auto_add_page", AUTO_ADD_PAGE) == "true") {
            // Set the page_url field to dirty so that the page will save
            // it and make sure that after a page has been viewed it will
            // be added to the pages database
            $this->SetValueDirty("page_url");
        }

        // Check to see if this page requires HTTPS
        if ($this->GetValue("require_ssl", "false") == "true" && $_SERVER["HTTPS"] != "on") {
            // Redirect to this page to the HTTPS
            $this->RedirectToSSL();
        }
        CMTKEvent::FireEvent("CMTKPage_Load", $this);

        // Deal properly with page permissions
        if ($this->AuthorizeUser() != true) {
            // This is needed for basic/digest auth
            // forms auth will redir to the login page
            $ret = $this->AuthenticateUser();            
            // Try to authorize again
            if ($this->AuthorizeUser() != true) {
                $fail_message = cmtk_GetConfigValue("authentication_failure_message",
                    AUTHENTICATION_FAILURE_MESSAGE);
                $this->SetValue("page_auth_error", $fail_message, false);
                return;
            }
        }

        // Process any commands
        $this->ProcessCommands();
        
        // Load the template file
        $this->GetTemplate();
        // RenderPage will be called by the obprocess function later        

        CMTKEvent::FireEvent("CMTKPage_LoadComplete", $this);
    }

    function __destruct() {
        // Save any changed values to the datasource
        $this->SaveObjectInfo();
    }

    function GetPageURL() {
        return $this->GetValue("page_url", "");
    }

    static function GetCurrentPage() {
        global $CMTK_CURRENT_PAGE;
        return $CMTK_CURRENT_PAGE;
    }

    function ProcessCommands() {
        // If a command has been sent, process it.
        /* Extract the following values:
         * Command Name - cmtk_command_name
         * Command Argument - cmtk_command_argument
         *
         */
        $cmd_name = cmtk_GetReqValue("cmtk_command_name", "");
        $cmd_arg = cmtk_GetReqValue("cmtk_command_arg", "");
        if ($cmd_name == "") { return; }

        $this->SetValue("cmtk_command_name", $cmd_name, false);
        $this->SetValue("cmtk_command_arg", $cmd_arg, false);
        
        CMTKEvent::FireEvent("CMTKPage_Command", $this);
    }

    function RenderPage($buffer) {
        // Fire the preRender event
        $this->SetValue("page_content", $buffer, false);
        $page_text = CMTKEvent::FireEvent("CMTKPage_PreRender", $this);
        $page_text = $this->GetValue("page_content", $buffer);

        // If the user is not authenticated, replace page content with the error message
        $page_auth_error = $this->GetValue("page_auth_error", "");
        if ($page_auth_error != "") {
            // Clear the error message so next time around it doesn't hit us
            $this->SetValue("page_auth_error", "", false);
            $page_text = $page_auth_error;
            //cmtk_LogError("Page Auth Failed", __FILE__, __LINE__);
        }

        // Make sure some values are present
        $this->SetValueIfNotExists("page_content", "", false);
        $this->SetValueIfNotExists("page_title", $this->GetPageTitle(), false);
        $this->SetValueIfNotExists("page_keywords", $this->GetPageKeywords(), false);
        $this->SetValueIfNotExists("page_description", $this->GetPageDescription(), false);
        $this->SetValueIfNotExists("page_ajax_include", $this->GetPageAjaxInclude(), false);
        $this->SetValueIfNotExists("page_menu", $this->GetPageMenu(), false);
               
        // Process the page, combine the template, deal with parts...
        // Put the page in the template        
        $page_text = str_replace("<PAGE_CONTENT>", $page_text,
            $this->GetValue("template_text", "<PAGE_CONTENT>"));

        // Put each value into the page
        foreach ($this->VALUES as $key=>$value) {
            //echo "Key: $key";
            //$page_text .= "Key: $key";
            $page_text = str_replace("<" . strtoupper($key) . ">", $value, $page_text);
        }

        // Process tags
        $page_text = $this->ProcessTags($page_text);

        // Fire the RenderComplete event
        $this->SetValue("page_content", $page_text, false);        
        $page_text = CMTKEvent::FireEvent("CMTKPage_RenderComplete", $this);
        $page_text = $this->GetValue("page_content", $page_text);

        // Display the errors if needed
        if (cmtk_GetConfigValue("display_errors", DISPLAY_ERRORS) == "true") {
            $page_text .= cmtk_GetErrorString();
        }

        // Show timing information if turned on
        if (cmtk_GetConfigValue("display_page_render_timing", DEFAULT_DISPLAY_PAGE_RENDER_TIMING) == "true") {
            $page_text .= cmtk_GetPageRenderTimingString();
        }
        
        return $page_text;
    }

    function ProcessTags($page_text) {
        // If tags are disabled, then return without processing tags
        $disabled = cmtk_GetConfigValue("disable_tags", DEFAULT_DISABLE_TAGS);
        if ($disabled == "true") { return false; }
        $disabled = $this->GetValue("disable_tags", DEFAULT_DISABLE_TAGS);
        if ($disabled == "true") { return false; }

        $doc = new DOMDocument("1.0", "UTF-8");
        $doc->preserveWhiteSpace = true;
        $doc->strictErrorChecking = false;
        $doc->resolveExternals = true;

        // loadHTML shows warnings for cmtk tags, supress the warnings
        if (@$doc->loadHTML($page_text) == false) {
            cmtk_LogError("Error parsing page", __FILE__, __LINE__);
            return $ret;
        }
        
        $parts = $doc->getElementsByTagName("cmtk_part");
        for ($index = 0; $index < $parts->length; $index++) {
            $part = $parts->item($index);
            $obj = $this->InitializePart($part);

            if ($obj === false) {
                // Object failed to create, replace with an error message
                // Replace the dom node with the error
                $err_node = $doc->createElement("b",cmtk_GetConfigValue("tags_error_message", DEFAULT_TAGS_ERROR_MESSAGE));
                if (is_object($part->parentNode)) {
                    $part->parentNode->replaceChild($err_node, $part);
                }
            }
        }

        // Call each parts process method
        foreach($this->PARTS as $part_name => $part) {
            if (is_object($part) && method_exists($part, "Process")) {
                $part->Process();
            }
        }

        // Call each parts render method        
        foreach($this->PARTS as $part_name => $part) {            
            $render_node = null;
            if (is_object($part) && method_exists($part, "Render")) {
                // Part has render method, have it render its output and return a string
                $render_string = "<span>" . $part->Render() . "</span>";
                // Wrap the string in a span node so we can replace the current node with it
                $tmp_doc = new DOMDocument();
                // loadHTML shows warnings for cmtk tags, suppress the warnings.
                if (@$tmp_doc->loadHTML($render_string) === false) {
                    cmtk_LogError("Invalid Part Render text ($part_name): $render_string", __FILE__, __LINE__);
                    $render_node = $doc->createElement("span", $render_string);
                } else {
                    //cmtk_LogError("Render: (" . $tmp_doc->saveHTML() . ")", __FILE__, __LINE__);
                    // DOMDocument wrapped our span tag in all the extra HTML junk,
                    // Grab just the body tag and return its child tag
                    // E.G. <body><span>...</span></body>
                    $body_node_list = $tmp_doc->getElementsByTagName("body");
                    foreach($body_node_list as $body_node) {
                        $render_node = $body_node->firstChild;
                    }
                    if ($render_node != null && $render_node !== false) {
                        // Need to import the node into the current document
                        // or it won't work later.
                        $dom_element = $part->GetDomElement();
                        $owner_doc = false;
                        if ($dom_element !== false) {
                            $owner_doc = $dom_element->ownerDocument;
                        }
                        if ($owner_doc === false) {
                            cmtk_LogError("Owner Doc Null!", __FILE__, __LINE__);
                        } else {
                            $render_node = $owner_doc->importNode($render_node, true);
                        }
                    }
                }
            } else {
                // No render method, create a dummy node to display the error
                $render_node = $doc->createElement("b",$part_name . " - " . cmtk_GetConfigValue("tags_no_render_method", DEFAULT_TAGS_NO_RENDER_METHOD_MESSAGE));
            }
            // Get the original dom element
            $dom_element = $part->GetDomElement();
            // Replace the node with the new render node
            if (is_object($dom_element->parentNode) && ($render_node instanceof DOMElement || $render_node instanceof DOMNode)) {
                $dom_element->parentNode->replaceChild($render_node, $dom_element);
            }
        }
        
        return $doc->saveHTML();
    }    

    function InitializePart($dom_node) {
        $ret = false;

        // Look in the list of PARTS for this object
        $part_name = $dom_node->getAttribute("name");
        if ($part_name == "") { return $ret; }
        if ($this->PartExists($part_name)) {
            $obj = $this->GetPart($part_name);
            // Re-assign the dom node so that the render works later
            $obj->InitializePart($dom_node);
            return $obj;
        }

        // Create an object for this part
        $object_name = $dom_node->getAttribute("object_name");
        if ($object_name == "") { return $ret; }

        // Use class_exists to attempt the auto load
        if( class_exists($object_name)) {
            $obj = new $object_name();
            if ($obj !== false) {
                $this->AddPart($part_name, $obj);
                $obj->InitializePart($dom_node);
                $ret = $obj;
            }
        }

        return $ret;
    }

    function GetPart($part_name) {
        return $this->PARTS[$part_name];
    }

    function PartExists($part_name) {
        $ret = false;
        if (isset($this->PARTS[$part_name])) {
            $ret = true;
        }
        return $ret;
    }

    function AddPart($part_name, $obj) {
        $this->PARTS[$part_name] = $obj;
    }

    function RemovePart($part_name) {
        if (isset($this->PARTS[$part_name])) {
            unset($this->PARTS[$part_name]);
        }
    }

    function AuthenticateUser() {
        $user = CMTKUser::GetCurrentUser();
        // Need to ask the user to login, should we
        // show forms auth or pop login box?        

        $auth_type = cmtk_GetConfigValue("authentication_type", AUTHENTICATION_TYPE);
        if ($auth_type == "forms") {
            // Redirect to login page
            $redir_url = cmtk_GetConfigValue("login_url", DEFAULT_LOGIN_URL);
            // add the current page url
            $redir_url .= "?redir_url=" . urlencode($_SERVER["SCRIPT_NAME"]);
            header("Location: $redir_url");
            exit;
        } elseif ($auth_type == "digest") {
            throw new Exception("Digest authentication not imlemented yet!");
            //http://us2.php.net/features.http-auth
        } else {    // default to basic authentication
            // Check to see if the user is trying to login
            if (isset($_SERVER['PHP_AUTH_USER']) && strlen($_SERVER['PHP_AUTH_USER']) > 0) {
                // Try to authenticate the user
                //TODO Auth User
                $user = CMTKUser::GetCurrentUser();
                if ($user->AuthenticateUser($_SERVER['PHP_AUTH_USER'], $_SERVER['PHP_AUTH_PW']) == true) {
                    // User authenticated
                    return true;
                }
                // User didn't authenticate, show login box again
            }

            // Show the basic login box
            $realm = cmtk_GetConfigValue("authentication_realm", AUTHENTICATION_REALM);
            $fail_message = cmtk_GetConfigValue("authentication_failure_message",
                AUTHENTICATION_FAILURE_MESSAGE);
            header("WWW-Authenticate: Basic realm=\"$realm\"");
            header("HTTP/1.0 401 Unauthorized");
            // Set the failure message so that it shows up instead of the page
            $this->SetValue("page_auth_error", $fail_message, false);
        }
    }

    function AuthorizeUser() {
        $user = CMTKUser::GetCurrentUser();        

        // Does the page require authorization?
        $view_users = $this->GetValue("view_users[]", array());
        $view_roles = $this->GetValue("view_roles[]", array());

        if (count($view_users) < 1 && count($view_roles) < 1) {
            // Page doesn't require any users or roles, viewable by everyone.
            return true;
        }

        // Make sure uesr has been authenticated
        if ($user->IsAuthenticated() != true) {
            return false;
        }

        // Check the users to see if we have a match
        if (in_array($user->GetUserName(), $view_users)) {
            return true;
        }

        // Check for match with email address
        if (in_array($user->GetEmailAddress(), $view_users)) {
            return true;
        }

        // See if the user is a member of one of the required roles
        foreach ($view_roles as $role) {
            if ($user->IsInRole($role) == true) {
                return true;
            }
        }
        
        //cmtk_LogError("User (" . $user->GetUserName() . ") not authorized to view $_SERVER[SCRIPT_NAME]",
        //    __FILE__, __LINE__);
        // If we get here, the user is not allowed to see this page
        return false;
    }

    private function GetTemplate() {
        global $app_template_directory;
        // Try grabbing the current template file
        $template_file = $this->GetValue("template_file", "");
        if ($template_file == "") {
            // No current template file, grab the default one
            $template_file = cmtk_GetConfigValue("default_template", DEFAULT_TEMPLATE);
        }
        
        if ($template_file == "") {
            // No template assigned, setup blank page
            $this->SetValue("template_text", "<PAGE_CONTENT>", false);
            cmtk_LogError("Invalid Template: No default template assigned.",
                __file__, __LINE__);
            return;
        }

        // Does this template file exist?
        if (!file_exists($template_file)) {
            // No file? Try adding directory to the front of it
            $orig_template_file = $template_file;
            $template_file = $app_template_directory . "/" . $template_file;
            if (!file_exists($template_file)) {
                // Still doesn't exist? Set to blank
                $this->SetValue("template_text", "<PAGE_CONTENT>", false);
                cmtk_LogError("Invalid Template: Invalid template file ($orig_template_file).",
                    __file__, __LINE__);
                return;
            }
        }

        // Include the template file
        ob_start();
        @include "$template_file";
        @$template_text = ob_get_contents();
        ob_end_clean();

        // Just in case, so a lack of template means that the page will still show up
        if ($template_text == "") { $template_text = "<PAGE_CONTENT>."; }

        $this->SetValue("template_text", $template_text, false);
        return;
    }
    
    public function SaveObjectInfo() {
        // Overrides parent function.
        // call parent function to save info
        parent::SaveObjectInfo();
        
        // Fire an event to signal the saving info is complete.
        CMTKEvent::FireEvent("CMTKPage_SaveInfoComplete", $this);
    }

    public function GetPageTitle() {
        $ret = $this->GetValue("page_title", cmtk_GetConfigValue("default_page_title", DEFAULT_PAGE_TITLE));
        return $ret;
    }

    public function GetPageKeywords() {
        $ret = $this->GetValue("page_keywords", cmtk_GetConfigValue("default_page_keywords", DEFAULT_PAGE_KEYWORDS));
        return $ret;
    }

    public function GetPageDescription() {
        $ret = $this->GetValue("page_description", cmtk_GetConfigValue("default_page_description", DEFAULT_PAGE_DESCRIPTION));
        return $ret;
    }

    public function GetPageAjaxInclude() {
        $ret = $this->GetValue("page_ajax_include", cmtk_GetConfigValue("default_page_ajax_include", DEFAULT_PAGE_AJAX_INCLUDE));
        return $ret;
    }

    public function GetPageMenu() {
        return "";
    }

    public function RedirectToSSL($redirect_url = "") {
        if ($redirect_url == "") {
            // Get the url from the page info
            $redirect_url = $this->GetValue("ssl_redirect_url", "");
        }
        if ($redirect_url == "") {
            $protocol = "http://";
            if ($_SERVER["HTTPS"] == "on") {
                $protocol = "https://";
            }
            $redirect_url = "$protocol$_SERVER[HTTP_HOST]$_SERVER[PHP_SELF]";
        }
        if ($redirect_url != "") {
            header("Location: $redirect_url");
            exit;
        }
    }
}

?>
